"""
HB_SelectionToCircle V1.3

Last Modified: Oct/26/2018
Works with CINEMA 4D R16-R20 and up.
Copyright: Holger Biebrach, www.c4dstuff.com

Name-US: HB_SelectionToCircle
Description-US: Makes current Selection Spherical/Radial [CTRL: Preserve Shape, SHIFT: Relax Spherical]

Select Points or Polygons and run the Script. The selection will be made Spherical.

Video Tutorial:
https://youtu.be/kQpLqrtGfBQ?t=21s

https://youtu.be/rAHc5mstOCg?t=446



Name-DE: HB_SelectionToCircle
Description-DE: Macht Selektion Spherisch/Kreisförmig

ChangeLog:

Jan/1/2016 V1.0
- Release Version

Nov/22/2016 V1.1
- Fixed issue where selection was scaled down

Oct/11/2018 v1.2
- New Highres Icon
- Fixed: Size Changes
- Uses Modeling Axis as Center
- Fixed: Compatible with Slidemode now
- Added CTRL to preserve Shape
- Added Relaxing of selected planar Polygons

Oct/26/2018 V1.3
- Fixed Issues with R20


"""

import c4d
from c4d import gui, documents, plugins, utils

def NormalMove(obj,amount):


    NMsettings = c4d.BaseContainer()                 # Settings
    NMsettings[c4d.MDATA_NORMALMOVE_VALUE] = amount

    utils.SendModelingCommand(command = c4d.ID_MODELING_NORMALMOVE_TOOL,
                                    list = [obj],
                                    mode = c4d.MODELINGCOMMANDMODE_ALL,
                                    bc = NMsettings,
                                    doc = doc)

def Relax(obj):



    c4d.utils.SendModelingCommand(command = c4d.MCOMMAND_SELECTSHRINK,
                                list = [obj],
                                mode = c4d.MODELINGCOMMANDMODE_POINTSELECTION,
                                bc = c4d.BaseContainer(),
                                doc = doc)


    ### Set Selection and Restrict

    seltag = obj.MakeTag(c4d.Tpointselection)
    seltag_bs = seltag.GetBaseSelect()
    obj.GetPointS().CopyTo(seltag_bs)

    seltag.SetName("tempsel")

    obj.InsertTag(seltag)
    doc.AddUndo(c4d.UNDOTYPE_NEW, seltag)

    #SMOOTHDEFORMER
    SmoothDeformer=c4d.BaseObject(1024529)
    SmoothDeformer[c4d.ID_CA_SMOOTHING_DEFORMER_OBJECT_STIFFNESS]=0
    SmoothDeformer[c4d.ID_CA_SMOOTHING_DEFORMER_OBJECT_STRENGTH]=100
    SmoothDeformer[c4d.ID_CA_SMOOTHING_DEFORMER_OBJECT_ITERATIONS]=100




    #RESTRICT DEFORMER



    restag = c4d.BaseTag(5683)
    restag[c4d.RESTRICTIONTAG_NAME_01]="tempsel"
    SmoothDeformer.InsertTag(restag)
    doc.AddUndo(c4d.UNDOTYPE_NEW, restag)

    SmoothDeformer.InsertUnder(obj)
    doc.AddUndo(c4d.UNDOTYPE_NEW, SmoothDeformer)

    CSsettings = c4d.BaseContainer()
    NewObj=utils.SendModelingCommand(command = c4d.MCOMMAND_CURRENTSTATETOOBJECT,
                                    list = [obj],
                                    mode = c4d.MODELINGCOMMANDMODE_ALL,
                                    bc = CSsettings,
                                    doc = doc,
                                    )


    NewObj[0].InsertAfter(obj)




    if NewObj[0].GetType()==c4d.Onull:
        NewObjChild=NewObj[0].GetDown()
        NewObj[0].Remove()

    else:
        NewObjChild=NewObj[0]


    NewObjChild.InsertAfter(obj)
    doc.AddUndo(c4d.UNDOTYPE_NEW, NewObjChild)

    #NewObjChild.SetMg(Pos)


    doc.AddUndo(c4d.UNDOTYPE_CHANGE_SMALL,NewObjChild)
    NewObjChild.SetBit(c4d.BIT_ACTIVE)

    ObjChild=NewObjChild.GetDown()
    if ObjChild:

        doc.AddUndo(c4d.UNDOTYPE_DELETE, ObjChild)
        ObjChild.Remove()

    doc.AddUndo(c4d.UNDOTYPE_DELETE, obj)
    obj.Remove()








def deletetags(obj): # Scan Tags on Object

    n       = 1

    tags    = obj.GetTags()

    for tag in tags:
        if not tag: continue
        tagtype = tag.GetType()

        if tag.GetName()=="tempsel":
            doc.AddUndo(c4d.UNDOTYPE_DELETE, tag)
            tag.Remove()
            c4d.EventAdd()


def GetModelingAxis(op):

    MAxis=op.GetModelingAxis(doc)

    return MAxis.off

def GetSelectionSize(op):

    global Spherical
    Spherical=False
    sel = op.GetPointS()
    points = [(i, point)
          for i, point in enumerate(op.GetAllPoints())
          if sel.IsSelected(i)]

    if not points:
      return

    selsizeminX = min(point[1].x for point in points)
    selsizemaxX = max(point[1].x for point in points)
    selsizeminY = min(point[1].y for point in points)
    selsizemaxY = max(point[1].y for point in points)
    selsizeminZ = min(point[1].z for point in points)
    selsizemaxZ = max(point[1].z for point in points)

    boundbox= c4d.Vector(selsizemaxX,selsizemaxY,selsizemaxZ)-c4d.Vector(selsizeminX,selsizeminY,selsizeminZ)
    selsize=boundbox.GetLength()
    sizeX=selsizemaxX-selsizeminX
    sizeY=selsizemaxY-selsizeminY
    sizeZ=selsizemaxZ-selsizeminZ
    if sizeX>0 and sizeY>0 and sizeZ>0:
        Spherical=True




    return selsize

def RemoveChildren(objlist):

    for obj in objlist:
        doc.AddUndo(c4d.UNDOTYPE_DELETE, obj)
        obj.Remove()



    c4d.EventAdd()
    return

def InsertChildren(objlist,root):


    for obj in objlist:

        obj.InsertUnder(root)
        doc.AddUndo(c4d.UNDOTYPE_NEW, obj)


    c4d.EventAdd()
    return



def RemoveChildren(objlist):

    for obj in objlist:
        doc.AddUndo(c4d.UNDOTYPE_DELETE, obj)
        obj.Remove()


    c4d.EventAdd()
    return


def main():


    selobj=doc.GetSelection()

    if len(selobj)>1 or len(selobj)==0 : #ERROR

        gui.MessageDialog("Select one Object!")
        return


    bc = c4d.BaseContainer()
    c4d.gui.GetInputState(c4d.BFM_INPUT_KEYBOARD,c4d.BFM_INPUT_CHANNEL,bc) #Modifier Key pressed?

    if bc[c4d.BFM_INPUT_QUALIFIER] ==2:
       CTRL=True
       if len(selobj)==2:
           gui.MessageDialog("Select one Object!")
           return
    else:
        CTRL=False

    if bc[c4d.BFM_INPUT_QUALIFIER] ==1: #SHIFT
        SHIFT=True
    else:
        SHIFT=False



    mode=doc.GetMode()


    obj=selobj[0]
    ObjChildren = obj.GetChildren()

    if doc.GetMode()==7: # is Polymode active? Run selction Command

        c4d.utils.SendModelingCommand(command = c4d.MCOMMAND_SELECTPOLYTOPOINT,
                                    list = [obj],
                                    mode = c4d.MODELINGCOMMANDMODE_POLYGONSELECTION,
                                    bc = c4d.BaseContainer(),
                                    doc = doc)

    if doc.GetMode()==6: # is EdgeMode active? Run selction Command

        ConvertSettings = c4d.BaseContainer()

        ConvertSettings[c4d.MDATA_CONVERTSELECTION_LEFT]=1
        ConvertSettings[c4d.MDATA_CONVERTSELECTION_RIGHT]=0

        c4d.utils.SendModelingCommand(command = c4d.MCOMMAND_CONVERTSELECTION,
                                    list = [obj],
                                    mode = c4d.MODELINGCOMMANDMODE_EDGESELECTION,
                                    bc = ConvertSettings,
                                    doc = doc)

    doc.SetMode(5)#pointmode

    ObjPos=obj.GetMg()
    selAxisPos=obj.GetModelingAxis(doc)
    newm    = obj.GetMg()

    selsize = GetSelectionSize(obj)

    #Factors for Spherical Radius or planar
    Factor1=0.707106781186534
    #0.707106781186534
    #0.707106781186548
    Factor2=0.816373180508274 # use for Sperical

    SphereDeformer=c4d.BaseObject(1001003)
    SphereDeformer[c4d.SPHERIFYOBJECT_STRENGTH]=1
    if Spherical== True:

        SphereDeformer[c4d.SPHERIFYOBJECT_RADIUS]= (selsize/2)*Factor2*Factor1 #Use Factors here
    else:
        SphereDeformer[c4d.SPHERIFYOBJECT_RADIUS]= (selsize/2)*Factor1 #Use Factors here




    c4d.CallCommand(12552) # Set Selection
    newseltag=doc.GetActiveTag()
    doc.AddUndo(c4d.UNDOTYPE_CHANGE_SMALL,newseltag)
    newseltag.SetName("tempsel")
    restag = c4d.BaseTag(5683)
    restag[c4d.RESTRICTIONTAG_NAME_01]="tempsel"
    SphereDeformer.InsertTag(restag)
    doc.AddUndo(c4d.UNDOTYPE_NEW, restag)

    SphereDeformer.InsertUnder(obj)
    doc.AddUndo(c4d.UNDOTYPE_NEW, SphereDeformer)
    SphereDeformer.SetMg(selAxisPos)


    Slidemode=False
    if obj.GetNext():

        if obj.GetNext().GetName()=="HB_SlideProxy":
            Slidemode=True

            NormalMove(obj,0.001)

    if CTRL : #KeepShapeMode

        NormalMove(obj,0.001)

        SlideObj=obj.GetClone()
        SlideObj.InsertAfter(obj)
        doc.AddUndo(c4d.UNDOTYPE_NEW, SlideObj)
        SlideObj.GetDown().Remove()




        doc.AddUndo(c4d.UNDOTYPE_CHANGE, SlideObj)
        SlideObj.SetMg(ObjPos)

        #SHRINKWRAP
        shrinkwrapdeformer=c4d.BaseObject(1019774)
        shrinkwrapdeformer[c4d.SHRINKWRAP_TARGETOBJECT]=SlideObj
        shrinkwrapdeformer[c4d.ID_BASEOBJECT_VISIBILITY_EDITOR]=1

        shrinkwrapdeformer.InsertAfter(SphereDeformer)
        doc.AddUndo(c4d.UNDOTYPE_NEW, shrinkwrapdeformer)
        shrinkwrapdeformer.Message(c4d.MSG_MENUPREPARE)






    CSsettings = c4d.BaseContainer()
    NewObj=utils.SendModelingCommand(command = c4d.MCOMMAND_CURRENTSTATETOOBJECT,
                                    list = [obj],
                                    mode = c4d.MODELINGCOMMANDMODE_EDGESELECTION,
                                    bc = CSsettings,
                                    doc = doc,
                                    )



    if NewObj[0].GetType()==c4d.Onull:
        NewObjChild=NewObj[0].GetDown()


    else:
        NewObjChild=NewObj[0]


    if CTRL:
       SlideObj.Remove()

    NewObjChild.InsertAfter(obj)
    doc.AddUndo(c4d.UNDOTYPE_NEW, NewObjChild)
    NewObjChild.SetMg(ObjPos)



    ObjChild=NewObj[0].GetDown()
    doc.AddUndo(c4d.UNDOTYPE_DELETE, ObjChild)
    ObjChild.Remove()
    doc.AddUndo(c4d.UNDOTYPE_DELETE, obj)
    obj.Remove()



    deletetags(NewObjChild)
    NewObjChildren=NewObjChild.GetChildren()

    RemoveChildren(NewObjChildren)

    InsertChildren(ObjChildren, NewObjChild)



    doc.AddUndo(c4d.UNDOTYPE_CHANGE_SMALL,NewObjChild)
    NewObjChild.SetBit(c4d.BIT_ACTIVE)
    NewObjChild.SetMg(ObjPos)

    if Slidemode==True:
        NormalMove(NewObjChild,-0.001)

    doc.SetMode(c4d.Mpoints)#pointmode
    if CTRL:
        doc.SetMode(mode)#OldMode
        return


    NewObjChildren = NewObjChild.GetChildren()
    if SHIFT or Spherical==False:
        Relax(NewObjChild)
        FinalObj=doc.GetActiveObject()
        FinalObj.SetMg(ObjPos)
        deletetags(FinalObj)
        InsertChildren(NewObjChildren,FinalObj)





    doc.SetMode(mode)#OldMode

    doc.EndUndo()



if __name__=='__main__':
    main()
    c4d.EventAdd()